/*
 * Decompiled with CFR 0.152.
 */
package org.openconcerto.utils;

import java.awt.Desktop;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.mail.Message;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import org.openconcerto.utils.CollectionUtils;
import org.openconcerto.utils.DesktopEnvironment;
import org.openconcerto.utils.OSFamily;
import org.openconcerto.utils.StreamUtils;
import org.openconcerto.utils.StringUtils;
import org.openconcerto.utils.io.PercentEncoder;
import org.openconcerto.utils.net.PlainMessage;
import org.openconcerto.utils.system.Powershell;

public abstract class EmailClient {
    private static EmailClient PREFERRED = null;
    private static final Pattern registryPattern = Pattern.compile("\\s+REG_SZ\\s+(.*)$", 8);
    private static final Pattern cmdLinePattern = Pattern.compile("(\"(.*?)\")|([^\\s\"]+)\\b");
    private static final Pattern wsPattern = Pattern.compile("[\\s&&[^ \t]]");
    private static final Pattern dictPattern;
    private static final String AppleMailBundleID = "com.apple.mail";
    private static final String ThunderbirdBundleID = "org.mozilla.thunderbird";
    public static final EmailClient MailTo;
    public static final EmailClient XDG;
    public static final EmailClient Outlook;
    public static final EmailClient AppleMail;
    private final EmailClientType type;

    static {
        String rolePattern = "(?:LSHandlerRoleAll\\s*=\\s*\"([\\w\\.]+)\";\\s*)?";
        dictPattern = Pattern.compile("\\{\\s*(?:LSHandlerRoleAll\\s*=\\s*\"([\\w\\.]+)\";\\s*)?LSHandlerURLScheme = mailto;\\s*(?:LSHandlerRoleAll\\s*=\\s*\"([\\w\\.]+)\";\\s*)?\\}");
        MailTo = new EmailClient(null){

            @Override
            public boolean composeNative(PlainMessage msg) {
                return false;
            }
        };
        XDG = new EmailClient(EmailClientType.XDG){

            @Override
            public boolean composeNative(PlainMessage msg) throws IOException, InterruptedException {
                String subject = msg.getSubject();
                String body = msg.getBody();
                Set<File> attachments = msg.getAttachments().keySet();
                ProcessBuilder pb = new ProcessBuilder("xdg-email");
                for (InternetAddress addr : msg.getRecipients(Message.RecipientType.CC)) {
                    pb.command().add("--cc");
                    pb.command().add(addr.toUnicodeString());
                }
                for (InternetAddress addr : msg.getRecipients(Message.RecipientType.BCC)) {
                    pb.command().add("--bcc");
                    pb.command().add(addr.toUnicodeString());
                }
                if (subject != null) {
                    pb.command().add("--subject");
                    pb.command().add(subject);
                }
                if (body != null) {
                    pb.command().add("--body");
                    pb.command().add(body);
                }
                for (File attachment : attachments) {
                    pb.command().add("--attach");
                    pb.command().add(attachment.getAbsolutePath());
                }
                pb.command().add(EmailClient.getJoinedTo(msg));
                pb.inheritIO();
                Process process = pb.start();
                process.getOutputStream().close();
                int returnCode = process.waitFor();
                if (returnCode != 0) {
                    throw new IllegalStateException("Non zero return code: " + returnCode);
                }
                return true;
            }
        };
        Outlook = new EmailClient(EmailClientType.Outlook){

            @Override
            protected boolean composeNative(PlainMessage msg) throws IOException, InterruptedException {
                return this.composePowershell(msg);
            }

            protected boolean composePowershell(PlainMessage msg) throws IOException, InterruptedException {
                String subject = msg.getSubject();
                String body = msg.getBody();
                Set<File> attachments = msg.getAttachments().keySet();
                Powershell pwsh = Powershell.getInstance();
                String template = new String(StreamUtils.read(EmailClient.class.getResourceAsStream("Outlook.powershell")), StandardCharsets.UTF_8);
                template = template.replace("@to@", pwsh.quoteArray(msg.getRecipients(Message.RecipientType.TO).stream().map(InternetAddress::toUnicodeString).collect(Collectors.toList())));
                template = template.replace("@cc@", pwsh.quoteArray(msg.getRecipients(Message.RecipientType.CC).stream().map(InternetAddress::toUnicodeString).collect(Collectors.toList())));
                template = template.replace("@bcc@", pwsh.quoteArray(msg.getRecipients(Message.RecipientType.BCC).stream().map(InternetAddress::toUnicodeString).collect(Collectors.toList())));
                template = template.replace("@subject@", pwsh.quote(subject == null ? "" : subject));
                template = template.replace("@attachments@", pwsh.quoteArray(attachments.stream().map(File::getAbsolutePath).collect(Collectors.toList())));
                template = template.replace("@fromAddr@", msg.getFrom() == null ? "$null" : pwsh.quote(3.asString(msg.getFrom())));
                ProcessBuilder pb = new ProcessBuilder(new String[0]);
                pb.command().add("powershell");
                pb.command().add("-EncodedCommand");
                pb.command().add(pwsh.getEncodedCommand(template));
                pb.inheritIO();
                pb.redirectInput(ProcessBuilder.Redirect.PIPE);
                Process process = pb.start();
                Throwable throwable = null;
                Object var10_12 = null;
                try (OutputStream in = process.getOutputStream();){
                    in.write(body.getBytes(StandardCharsets.UTF_8));
                }
                catch (Throwable throwable2) {
                    if (throwable == null) {
                        throwable = throwable2;
                    } else if (throwable != throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                int returnCode = process.waitFor();
                if (returnCode != 0) {
                    throw new IllegalStateException("Non zero return code: " + returnCode + "\nCommande : " + CollectionUtils.join(pb.command(), " "));
                }
                return true;
            }
        };
        AppleMail = new EmailClient(EmailClientType.AppleMail){

            private final void addRecipients(PlainMessage msg, Message.RecipientType type, String asType, PrintStream w) {
                for (InternetAddress addr : msg.getRecipients(type)) {
                    String name = StringUtils.isEmpty(addr.getPersonal()) ? "" : ", " + EmailClient.createASParam("name", addr.getPersonal());
                    w.println(" tell theMessage to make new " + asType + " with properties {" + EmailClient.createASParam("address", addr.getAddress()) + name + "}");
                }
            }

            @Override
            protected boolean composeNative(PlainMessage msg) throws IOException, InterruptedException {
                String subject = msg.getSubject();
                String body = msg.getBody();
                Set<File> attachments = msg.getAttachments().keySet();
                Process process = Runtime.getRuntime().exec(new String[]{"osascript"});
                PrintStream w = new PrintStream(new BufferedOutputStream(process.getOutputStream()));
                w.println("tell application id \"com.apple.mail\"");
                w.println(" set theMessage to make new outgoing message with properties {" + EmailClient.getAppleMailParam(msg.getFrom(), subject, body) + "}");
                this.addRecipients(msg, Message.RecipientType.TO, "to recipient", w);
                this.addRecipients(msg, Message.RecipientType.CC, "cc recipient", w);
                this.addRecipients(msg, Message.RecipientType.BCC, "bcc recipient", w);
                for (File attachment : attachments) {
                    w.println(" tell content of theMessage to make new attachment with properties {" + EmailClient.createASParam("file name", attachment.getAbsolutePath()) + "} at after last paragraph");
                }
                w.println("end tell");
                w.close();
                if (w.checkError()) {
                    throw new IOException();
                }
                int returnCode = process.waitFor();
                if (returnCode != 0) {
                    throw new IllegalStateException("Non zero return code: " + returnCode);
                }
                return true;
            }
        };
    }

    public static final EmailClient getPreferred() throws IOException {
        if (PREFERRED == null) {
            PREFERRED = EmailClient.findPreferred();
            assert (PREFERRED != null);
        }
        return PREFERRED;
    }

    public static final void resetPreferred() {
        PREFERRED = null;
    }

    protected static String asString(InternetAddress addr) {
        return addr.toUnicodeString();
    }

    private static final String createEncodedParam(String name, String value) {
        return String.valueOf(name) + "=" + PercentEncoder.encode(value, StringUtils.UTF8);
    }

    private static final String createASParam(String name, String value) {
        return String.valueOf(name) + ":" + StringUtils.doubleQuote(value);
    }

    private static final String createVBParam(String name, String value) {
        String switchName = "/" + name + ":";
        if (value == null || value.length() == 0) {
            return switchName;
        }
        String encoded = PercentEncoder.encodeUTF16(value);
        assert (encoded.indexOf(34) < 0) : "Encoded contains a double quote, this will confuse cscript";
        return String.valueOf(switchName) + '\"' + encoded + '\"';
    }

    protected static PlainMessage createMsg(String to, String subject, String body, File ... attachments) throws IOException {
        PlainMessage res = new PlainMessage();
        try {
            res.setRecipients(Message.RecipientType.TO, to);
        }
        catch (AddressException e) {
            e.printStackTrace();
        }
        res.setSubject(subject);
        res.setBody(body);
        if (attachments.length > 0) {
            File[] fileArray = attachments;
            int n = attachments.length;
            int n2 = 0;
            while (n2 < n) {
                File file = fileArray[n2];
                res.getAttachments().put(file, file.getName());
                ++n2;
            }
        }
        return res;
    }

    private static String getCommaSeparatedRecipients(Collection<? extends InternetAddress> addresses, Function<? super InternetAddress, String> toString) {
        return addresses.stream().map(toString).collect(Collectors.joining(","));
    }

    private static String getJoinedTo(PlainMessage msg) {
        return EmailClient.getCommaSeparatedRecipients(msg.getRecipients(Message.RecipientType.TO), InternetAddress::toUnicodeString);
    }

    private static void addEncodedRecipients(List<String> l, String paramName, PlainMessage msg, Message.RecipientType type) {
        List<InternetAddress> recipients = msg.getRecipients(type);
        if (recipients.isEmpty()) {
            return;
        }
        l.add(EmailClient.createEncodedParam(paramName, EmailClient.getCommaSeparatedRecipients(recipients, InternetAddress::toUnicodeString)));
    }

    public static final URI getMailToURI(String to, String subject, String body, File ... attachments) throws IOException {
        return EmailClient.getMailToURI(EmailClient.createMsg(to, subject, body, attachments));
    }

    public static final URI getMailToURI(PlainMessage msg) throws IOException {
        String subject = msg.getSubject();
        String body = msg.getBody();
        Set<File> attachments = msg.getAttachments().keySet();
        String encodedTo = PercentEncoder.encode(EmailClient.getJoinedTo(msg), StringUtils.UTF8);
        ArrayList<String> l = new ArrayList<String>(4);
        EmailClient.addEncodedRecipients(l, "cc", msg, Message.RecipientType.CC);
        EmailClient.addEncodedRecipients(l, "bcc", msg, Message.RecipientType.BCC);
        if (subject != null) {
            l.add(EmailClient.createEncodedParam("subject", subject));
        }
        if (body != null) {
            l.add(EmailClient.createEncodedParam("body", body));
        }
        for (File attachment : attachments) {
            l.add(EmailClient.createEncodedParam("attachment", attachment.getAbsolutePath()));
        }
        String query = CollectionUtils.join(l, "&");
        try {
            return new URI("mailto:" + encodedTo + "?" + query);
        }
        catch (URISyntaxException e) {
            throw new IOException("Couldn't create mailto URI", e);
        }
    }

    private static final String getTBParam(PlainMessage msg) {
        String subject = msg.getSubject();
        String body = msg.getBody();
        Set<File> attachments = msg.getAttachments().keySet();
        ArrayList<String> l = new ArrayList<String>(4);
        if (msg.getFrom() != null) {
            l.add(EmailClient.createEncodedParam("from", EmailClient.asString(msg.getFrom())));
        }
        EmailClient.addEncodedRecipients(l, "to", msg, Message.RecipientType.TO);
        EmailClient.addEncodedRecipients(l, "cc", msg, Message.RecipientType.CC);
        EmailClient.addEncodedRecipients(l, "bcc", msg, Message.RecipientType.BCC);
        if (subject != null) {
            l.add(EmailClient.createEncodedParam("subject", subject));
        }
        if (body != null) {
            l.add(EmailClient.createEncodedParam("body", body));
        }
        ArrayList<String> urls = new ArrayList<String>(attachments.size());
        for (File attachment : attachments) {
            String rawPath = attachment.toURI().getRawPath();
            String tbURL = String.valueOf(rawPath.startsWith("//") ? "file:///" : "file://") + rawPath;
            urls.add(tbURL);
        }
        l.add(EmailClient.createEncodedParam("attachment", CollectionUtils.join(urls, ",")));
        return DesktopEnvironment.getDE().quoteParamForExec(CollectionUtils.join(l, ","));
    }

    private static final String getAppleMailParam(InternetAddress from, String subject, String body) {
        ArrayList<String> l = new ArrayList<String>(3);
        l.add("visible:true");
        if (from != null) {
            l.add(EmailClient.createASParam("sender", EmailClient.asString(from)));
        }
        if (subject != null) {
            l.add(EmailClient.createASParam("subject", subject));
        }
        if (body != null) {
            l.add(EmailClient.createASParam("content", body));
        }
        return CollectionUtils.join(l, ", ");
    }

    private static String[] tbCommand(String cmdLine, String toReplace, PlainMessage msg) {
        String composeArg = EmailClient.getTBParam(msg);
        ArrayList<String> arguments = new ArrayList<String>();
        Matcher cmdMatcher = cmdLinePattern.matcher(cmdLine);
        while (cmdMatcher.find()) {
            String quoted = cmdMatcher.group(2);
            String unquoted = cmdMatcher.group(3);
            assert (quoted == null ^ unquoted == null) : "Both quoted and unquoted, or neither quoted nor quoted: " + quoted + " and " + unquoted;
            String arg = quoted != null ? quoted : unquoted;
            boolean replace = arg.equals(toReplace);
            if (replace && !arguments.contains("-compose")) {
                arguments.add("-compose");
            }
            arguments.add(replace ? composeArg : arg);
        }
        return arguments.toArray(new String[arguments.size()]);
    }

    public final void compose(String to, String subject, String body, File ... attachments) throws IOException, InterruptedException {
        this.compose(EmailClient.createMsg(to, subject, body, attachments));
    }

    public final void compose(PlainMessage msg) throws IOException, InterruptedException {
        for (File attachment : msg.getAttachments().keySet()) {
            if (attachment.exists()) continue;
            throw new IOException("Attachment doesn't exist: '" + attachment.getAbsolutePath() + "'");
        }
        boolean handled = this.composeNative(msg);
        if (!handled) {
            URI mailto = EmailClient.getMailToURI(msg);
            Desktop.getDesktop().mail(mailto);
        }
    }

    private static String cmdSubstitution(String ... args) throws IOException {
        return DesktopEnvironment.cmdSubstitution(Runtime.getRuntime().exec(args));
    }

    private static EmailClient findPreferred() throws IOException {
        DesktopEnvironment de = DesktopEnvironment.getDE();
        if (de instanceof DesktopEnvironment.Windows) {
            String out = EmailClient.cmdSubstitution("reg", "query", "HKEY_CLASSES_ROOT\\mailto\\shell\\open\\command");
            Matcher registryMatcher = registryPattern.matcher(out);
            if (registryMatcher.find()) {
                String cmdLine = registryMatcher.group(1);
                if (cmdLine.contains("thunderbird")) {
                    return new ThunderbirdCommandLine(cmdLine, "%1");
                }
                if (cmdLine.toLowerCase().contains("outlook")) {
                    return Outlook;
                }
            }
        } else if (de instanceof DesktopEnvironment.Mac) {
            String bundleID;
            String dict = EmailClient.cmdSubstitution("defaults", "read", "com.apple.LaunchServices", "LSHandlers");
            Matcher dictMatcher = dictPattern.matcher(dict);
            if (dictMatcher.find()) {
                String before = dictMatcher.group(1);
                String after = dictMatcher.group(2);
                assert (before == null ^ after == null) : "Both before and after, or neither before nor after: " + before + " and " + after;
                bundleID = before != null ? before : after;
            } else {
                bundleID = AppleMailBundleID;
            }
            if (bundleID.equals(AppleMailBundleID)) {
                return AppleMail;
            }
            if (bundleID.equals(ThunderbirdBundleID)) {
                File appDir = ((DesktopEnvironment.Mac)de).getAppDir(bundleID);
                File exe = new File(appDir, "Contents/MacOS/thunderbird-bin");
                return new ThunderbirdPath(exe);
            }
        } else {
            if (de instanceof DesktopEnvironment.Gnome) {
                String cmdLine;
                if (de.getVersion().startsWith("2.") && (cmdLine = EmailClient.cmdSubstitution("gconftool", "-g", "/desktop/gnome/url-handlers/mailto/command")).contains("thunderbird")) {
                    return new ThunderbirdCommandLine(cmdLine, "%s");
                }
                return XDG;
            }
            if (de instanceof DesktopEnvironment.KDE) {
                return XDG;
            }
            if (de instanceof DesktopEnvironment.XFCE) {
                return XDG;
            }
            if (OSFamily.getInstance() instanceof OSFamily.Unix) {
                return XDG;
            }
        }
        return MailTo;
    }

    public EmailClient(EmailClientType type) {
        this.type = type;
    }

    public final EmailClientType getType() {
        return this.type;
    }

    protected abstract boolean composeNative(PlainMessage var1) throws IOException, InterruptedException;

    public String toString() {
        EmailClientType t = this.getType();
        return t == null ? "mailto" : t.toString();
    }

    public static final void main(String[] args) throws Exception {
        if (args.length == 1 && "--help".equals(args[0])) {
            System.out.println("Usage: java [-Dparam=value] " + EmailClient.class.getName() + " [EmailClientType args]");
            System.out.println("\tEmailClientType: mailto or " + Arrays.asList(EmailClientType.values()));
            System.out.println("\tparam: to, subject, body, files (seprated by ',' double it to escape)");
            return;
        }
        EmailClient client = EmailClient.createFromString(args);
        System.out.println("Using " + (args.length == 0 ? "preferred" : "passed") + " client : " + client);
        String from = System.getProperty("from");
        String to = System.getProperty("to", "Pierre Dupond <p.dupond@example.com>, p.dupont@server.com");
        String cc = System.getProperty("cc", "Paul Dupond <paul.dupond@example.com>, paul.dupont@server.com");
        String bcc = System.getProperty("bcc");
        String subject = System.getProperty("subject", "Suj\u00e9 \u20ac du courrier ',to='&;\\<> \"autre'\n2nd line");
        String body = System.getProperty("body", "Bonjour,\n\tsingle ' double \" backslash(arri\u00e8re) \\ slash /");
        String filesPath = System.getProperty("files");
        String[] paths = filesPath == null || filesPath.length() == 0 ? new String[]{} : filesPath.split("(?<!,),(?!,)");
        File[] f = new File[paths.length];
        int i = 0;
        while (i < f.length) {
            f[i] = new File(paths[i].replace(",,", ","));
            ++i;
        }
        PlainMessage msg = EmailClient.createMsg(to, subject, body, f);
        msg.setFrom(from);
        msg.setRecipients(Message.RecipientType.CC, cc);
        msg.setRecipients(Message.RecipientType.BCC, bcc);
        client.compose(msg);
    }

    private static final EmailClient createFromString(String ... args) throws IOException {
        if (args.length == 0) {
            return EmailClient.getPreferred();
        }
        if ("mailto".equals(args[0])) {
            return MailTo;
        }
        EmailClientType t = EmailClientType.valueOf(args[0]);
        switch (t) {
            case XDG: {
                return XDG;
            }
            case Outlook: {
                return Outlook;
            }
            case AppleMail: {
                return AppleMail;
            }
            case Thunderbird: {
                Thunderbird res = null;
                if (args.length == 2) {
                    File exe = new File(args[1]);
                    res = Thunderbird.createFromExe(exe);
                    if (res == null) {
                        throw new IOException("Invalid exe : " + exe);
                    }
                } else if (args.length == 3) {
                    res = Thunderbird.createFromCommandLine(args[1], args[2]);
                } else {
                    throw new IllegalArgumentException((Object)((Object)t) + " needs 1 or 2 arguments");
                }
                return res;
            }
        }
        throw new IllegalStateException("Unknown type " + (Object)((Object)t));
    }

    public static enum EmailClientType {
        Thunderbird,
        AppleMail,
        Outlook,
        XDG;

    }

    public static abstract class Thunderbird
    extends EmailClient {
        public static Thunderbird createFromExe(File exe) {
            if (exe == null) {
                throw new NullPointerException();
            }
            if (!exe.isFile()) {
                return null;
            }
            return new ThunderbirdPath(exe);
        }

        public static Thunderbird createFromCommandLine(String cmdLine, String toReplace) {
            return new ThunderbirdCommandLine(cmdLine, toReplace);
        }

        protected Thunderbird() {
            super(EmailClientType.Thunderbird);
        }

        @Override
        public String toString() {
            return this.getClass().getSimpleName();
        }
    }

    private static final class ThunderbirdCommandLine
    extends Thunderbird {
        private final String cmdLine;
        private final String toReplace;

        private ThunderbirdCommandLine(String cmdLine, String toReplace) {
            this.cmdLine = cmdLine;
            this.toReplace = toReplace;
        }

        @Override
        protected boolean composeNative(PlainMessage msg) throws IOException {
            Runtime.getRuntime().exec(EmailClient.tbCommand(this.cmdLine, this.toReplace, msg));
            return true;
        }

        @Override
        public String toString() {
            return String.valueOf(super.toString()) + " " + this.cmdLine;
        }
    }

    private static final class ThunderbirdPath
    extends Thunderbird {
        private final File exe;

        private ThunderbirdPath(File exe) {
            this.exe = exe;
        }

        @Override
        protected boolean composeNative(PlainMessage msg) throws IOException {
            String composeArg = EmailClient.getTBParam(msg);
            Runtime.getRuntime().exec(new String[]{this.exe.getPath(), "-compose", composeArg});
            return true;
        }

        @Override
        public String toString() {
            return String.valueOf(super.toString()) + " " + this.exe;
        }
    }
}

