/*
 * Decompiled with CFR 0.152.
 */
package org.openconcerto.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.SocketException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.openconcerto.utils.CollectionUtils;
import org.openconcerto.utils.DesktopEnvironment;
import org.openconcerto.utils.FileUtils;
import org.openconcerto.utils.Log;
import org.openconcerto.utils.OSFamily;
import org.openconcerto.utils.RTInterruptedException;
import org.openconcerto.utils.StringUtils;
import org.openconcerto.utils.cc.ITransformer;

public abstract class Platform {
    private static final int PING_TIMEOUT = 250;
    private static final Platform LINUX = new UnixPlatform(){

        @Override
        protected String getSelfProcessSymlink() {
            return "/proc/self";
        }

        @Override
        public PingResult ping(InetAddress host, PingBuilder pingBuilder, int routingTableIndex) throws IOException {
            if (routingTableIndex > 0) {
                throw new UnsupportedOperationException("On Linux, choosing a different routing table requires changing the system policy");
            }
            ArrayList<String> command = new ArrayList<String>(16);
            command.add("ping");
            int totalCount = pingBuilder.getTotalCount();
            command.add("-c");
            command.add(String.valueOf(totalCount));
            if (pingBuilder.getWaitTime() > 0) {
                command.add("-W");
                int timeInSeconds = pingBuilder.getWaitTime() / 1000;
                command.add(String.valueOf(Math.max(timeInSeconds, 1)));
            }
            command.add("-M");
            command.add(pingBuilder.isDontFragment() ? "do" : "dont");
            if (pingBuilder.getLength() > 0) {
                command.add("-s");
                command.add(String.valueOf(pingBuilder.getLength()));
            }
            if (pingBuilder.getTTL() > 0) {
                command.add("-t");
                command.add(String.valueOf(pingBuilder.getTTL()));
            }
            if (pingBuilder.getSourceAddress() != null) {
                command.add("-I");
                command.add(pingBuilder.getSourceAddress());
            }
            command.add(host.getHostAddress());
            return this.ping("out=$(" + CollectionUtils.join(command, " ") + ") ; grep -c ttl= <<< \"$out\" ; tail -n1 <<< \"$out\"", totalCount, pingBuilder.getRequiredReplies());
        }
    };
    private static final Platform FREEBSD = new UnixPlatform(){

        @Override
        protected String getSelfProcessSymlink() {
            return "/proc/curproc";
        }

        @Override
        public PingResult ping(InetAddress host, PingBuilder pingBuilder, int routingTableIndex) throws IOException {
            ArrayList<String> command = new ArrayList<String>(16);
            command.add("setfib");
            command.add(String.valueOf(routingTableIndex));
            command.add("ping");
            int totalCount = pingBuilder.getTotalCount();
            command.add("-c");
            command.add(String.valueOf(totalCount));
            if (pingBuilder.getWaitTime() > 0) {
                command.add("-W");
                command.add(String.valueOf(pingBuilder.getWaitTime()));
            }
            if (pingBuilder.isDontFragment()) {
                command.add("-D");
            }
            if (pingBuilder.getLength() > 0) {
                command.add("-s");
                command.add(String.valueOf(pingBuilder.getLength()));
            }
            if (pingBuilder.getTTL() > 0) {
                command.add("-m");
                command.add(String.valueOf(pingBuilder.getTTL()));
            }
            if (pingBuilder.getSourceAddress() != null) {
                command.add("-S");
                command.add(pingBuilder.getSourceAddress());
            }
            command.add(host.getHostAddress());
            return this.ping("out=$(" + CollectionUtils.join(command, " ") + ") ; grep -c ttl= <<< \"$out\" ; tail -n1 <<< \"$out\"", totalCount, pingBuilder.getRequiredReplies());
        }
    };
    private static final Platform MACOS = new UnixPlatform(){

        @Override
        protected String getSelfProcessSymlink() {
            return null;
        }

        @Override
        protected PingResult ping(InetAddress host, PingBuilder pingBuilder, int routingTableIndex) throws IOException {
            return FREEBSD.ping(host, pingBuilder, routingTableIndex);
        }
    };
    private static final Platform CYGWIN = new CygwinPlatform();

    public static final Platform getInstance() {
        OSFamily os = OSFamily.getInstance();
        if (os == OSFamily.Windows) {
            return CYGWIN;
        }
        if (os == OSFamily.FreeBSD) {
            return FREEBSD;
        }
        if (os == OSFamily.Mac) {
            return MACOS;
        }
        return LINUX;
    }

    public abstract boolean supportsPID();

    public abstract boolean isRunning(int var1) throws IOException;

    public abstract String getPath(File var1);

    public final String getPID() throws IOException {
        try {
            Class<?> phClass = Class.forName("java.lang.ProcessHandle");
            Object ph = phClass.getMethod("current", new Class[0]).invoke(null, new Object[0]);
            return ((Number)phClass.getMethod("pid", new Class[0]).invoke(ph, new Object[0])).toString();
        }
        catch (ClassNotFoundException phClass) {
        }
        catch (Exception e) {
            throw new IOException("Couldn't get PID", e);
        }
        return this.getPreJava9PID();
    }

    protected String getPreJava9PID() throws IOException {
        Process p = this.eval("echo -n $PPID");
        Throwable throwable = null;
        Object var3_4 = null;
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(p.getInputStream()));){
            return reader.readLine();
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    public final String tail(File f, int n) throws IOException {
        Process p = Runtime.getRuntime().exec(new String[]{"tail", "-n" + n, this.getPath(f)});
        return DesktopEnvironment.cmdSubstitution(p);
    }

    protected abstract String getBash();

    public final void ln_s(File f1, File f2) throws IOException {
        FileUtils.ln(f1, f2);
    }

    public boolean isSymLink(File f) throws IOException {
        return this.exitStatus(Runtime.getRuntime().exec(new String[]{"test", "-L", f.getAbsolutePath()})) == 0;
    }

    public File getNativeSymlinkFile(File dir) {
        return dir;
    }

    public abstract String readLink(File var1) throws IOException;

    public final boolean exists(File f) throws IOException {
        return this.exitStatus(Runtime.getRuntime().exec(new String[]{"test", "-e", f.getAbsolutePath()})) == 0;
    }

    public final void append(File f1, File f2) throws IOException {
        String c = "cat '" + f1.getAbsolutePath() + "' >> '" + f2.getAbsolutePath() + "'";
        this.waitForSuccess(this.eval(c), "append");
    }

    public Process cp_l(File src, File dest) throws IOException {
        return Runtime.getRuntime().exec(new String[]{"cp", "-prl", src.getAbsolutePath(), dest.getAbsolutePath()});
    }

    public final boolean ping(InetAddress host) throws IOException {
        return this.ping(host, 250);
    }

    public abstract boolean ping(InetAddress var1, int var2) throws IOException;

    public final PingBuilder createPingBuilder() {
        return new PingBuilder(this);
    }

    protected abstract PingResult ping(InetAddress var1, PingBuilder var2, int var3) throws IOException;

    protected abstract BigDecimal parsePingAverageRT(String var1);

    protected final PingResult ping(String command, int totalCount, int requiredCount) throws IOException {
        if (requiredCount <= 0) {
            requiredCount = totalCount;
        }
        Process proc = this.evalPB(command).redirectErrorStream(false).start();
        String output = DesktopEnvironment.cmdSubstitution(proc);
        try {
            this.waitForSuccess(proc, "ping");
            List<String> countAndLastLine = StringUtils.splitIntoLines(output);
            if (countAndLastLine.size() != 2) {
                throw new IllegalStateException("Not 2 lines in " + countAndLastLine);
            }
            int replied = Integer.parseInt(countAndLastLine.get(0));
            assert (replied <= totalCount);
            BigDecimal averageRTT = replied == 0 ? null : this.parsePingAverageRT(countAndLastLine.get(1).trim());
            return new PingResult(totalCount, replied, requiredCount, averageRTT);
        }
        catch (Exception e) {
            throw new IllegalStateException("Couldn't use output :<<<\n" + output + "\n<<<", e);
        }
    }

    public final Process eval(String s) throws IOException {
        return this.evalPB(s).start();
    }

    public final ProcessBuilder evalPB(String s) throws IOException {
        return new ProcessBuilder(this.getBash(), "-c", s);
    }

    public final int exitStatus(Process p) {
        return this.exitStatus(p, null);
    }

    public final int exitStatus(Process p, String name) {
        try {
            return p.waitFor();
        }
        catch (InterruptedException e) {
            throw new RTInterruptedException("Interrupted while waiting for" + (name == null ? "" : " '" + name + "'") + " process", e);
        }
    }

    public final void waitForSuccess(Process p, String name) {
        int exitStatus = this.exitStatus(p, name);
        if (exitStatus != 0) {
            throw new IllegalStateException(String.valueOf(name) + " unsuccessful : " + exitStatus);
        }
    }

    public abstract boolean isAdmin() throws IOException;

    public static String toCygwinPath(File dir) {
        List<File> ancestors = Platform.getAncestors(dir);
        String root = ancestors.get(0).getPath();
        List<File> rest = ancestors.subList(1, ancestors.size());
        return "/cygdrive/" + root.charAt(0) + "/" + CollectionUtils.join(rest, "/", new ITransformer<File, String>(){

            @Override
            public String transformChecked(File f) {
                return f.getName();
            }
        });
    }

    public static List<File> getAncestors(File f) {
        File abs;
        File current = abs = f.getAbsoluteFile();
        ArrayList<File> res = new ArrayList<File>();
        while (current != null) {
            res.add(0, current);
            current = current.getParentFile();
        }
        return res;
    }

    private static final class CygwinPlatform
    extends Platform {
        private static final Pattern PING_STATS_PATTERN = Pattern.compile("^\\p{Blank}*\\p{Alpha}+ = ([0-9\\.]+)ms, \\p{Alpha}+ = ([0-9\\.]+)ms, \\p{Alpha}+ = ([0-9\\.]+)ms$");

        private CygwinPlatform() {
        }

        @Override
        public boolean supportsPID() {
            return false;
        }

        @Override
        public boolean isRunning(int pid) throws IOException {
            Process p = this.eval("export PATH=$PATH:/usr/bin ; test $(ps -sW -p " + pid + " | wc -l) -eq 2 ");
            return this.exitStatus(p) == 0;
        }

        @Override
        protected String getBash() {
            return "bash.exe";
        }

        @Override
        public String readLink(File f) throws IOException {
            Process p = Runtime.getRuntime().exec(new String[]{"readshortcut.exe", "-w", f.getAbsolutePath()});
            return DesktopEnvironment.cmdSubstitution(p).trim();
        }

        @Override
        public String getPath(File f) {
            return CygwinPlatform.toCygwinPath(f);
        }

        @Override
        public boolean ping(InetAddress host, int timeout) throws IOException {
            int exit = this.exitStatus(Runtime.getRuntime().exec("ping -n 1 -w " + timeout + " " + host.getHostAddress()), "ping");
            return exit == 0;
        }

        @Override
        public PingResult ping(InetAddress host, PingBuilder pingBuilder, int routingTableIndex) throws IOException {
            if (routingTableIndex > 0) {
                throw new UnsupportedOperationException("Only one routing table on Windows");
            }
            ArrayList<String> command = new ArrayList<String>(16);
            command.add("ping");
            int totalCount = pingBuilder.getTotalCount();
            command.add("-n");
            command.add(String.valueOf(totalCount));
            if (pingBuilder.getWaitTime() > 0) {
                command.add("-w");
                command.add(String.valueOf(pingBuilder.getWaitTime()));
            }
            if (pingBuilder.isDontFragment()) {
                command.add("-f");
            }
            if (pingBuilder.getLength() > 0) {
                command.add("-l");
                command.add(String.valueOf(pingBuilder.getLength()));
            }
            if (pingBuilder.getTTL() > 0) {
                command.add("-i");
                command.add(String.valueOf(pingBuilder.getTTL()));
            }
            if (pingBuilder.getSourceAddress() != null) {
                command.add("-S");
                command.add(pingBuilder.getSourceAddress());
            }
            command.add(host.getHostAddress());
            return this.ping("tmpF=$(mktemp) && " + CollectionUtils.join(command, " ") + " > $tmpF ; grep -c \"TTL=\" < $tmpF ; tail -n1 $tmpF; rm \"$tmpF\"", totalCount, pingBuilder.getRequiredReplies());
        }

        @Override
        protected BigDecimal parsePingAverageRT(String statsLine) {
            Matcher m = PING_STATS_PATTERN.matcher(statsLine);
            if (!m.matches()) {
                throw new IllegalArgumentException("Not matching " + PING_STATS_PATTERN + " : " + statsLine);
            }
            return new BigDecimal(m.group(3));
        }

        @Override
        public boolean isAdmin() throws IOException {
            return this.exitStatus(this.eval("id -G | egrep '\\b(544|114)\\b'")) == 0;
        }

        @Override
        public boolean isSymLink(File f) throws IOException {
            return this.getNativeSymlinkFile(f).exists() || super.isSymLink(f);
        }

        @Override
        public File getNativeSymlinkFile(File dir) {
            return FileUtils.addSuffix(dir, ".lnk");
        }
    }

    public static final class Ping {
        public static void main(String[] args) throws IOException {
            int totalCount = Integer.parseInt(System.getProperty("count", "4"));
            String srcAddr = System.getProperty("srcAddr");
            System.out.println(Platform.getInstance().createPingBuilder().setTotalCount(totalCount).setSourceAddress(srcAddr).execute(InetAddress.getByName(args[0])));
        }
    }

    public static final class PingBuilder {
        private final Platform platform;
        private int waitTime = 4000;
        private boolean dontFragment = false;
        private int length = -1;
        private String srcAddr = null;
        private int ttl = -1;
        private int totalCount = 4;
        private int requiredReplies = -1;

        PingBuilder(Platform p) {
            this.platform = p;
        }

        public final PingBuilder setWaitTime(int waitTime) {
            this.waitTime = waitTime;
            return this;
        }

        public final int getWaitTime() {
            return this.waitTime;
        }

        public final boolean isDontFragment() {
            return this.dontFragment;
        }

        public final PingBuilder setDontFragment(boolean dontFragment) {
            this.dontFragment = dontFragment;
            return this;
        }

        public final int getLength() {
            return this.length;
        }

        public final PingBuilder setLength(int length) {
            this.length = length;
            return this;
        }

        public String getSourceAddress() {
            return this.srcAddr;
        }

        public PingBuilder setSourceAddress(String srcAddr) {
            this.srcAddr = srcAddr;
            return this;
        }

        public final int getTTL() {
            return this.ttl;
        }

        public final PingBuilder setTTL(int ttl) {
            this.ttl = ttl;
            return this;
        }

        public final int getTotalCount() {
            return this.totalCount;
        }

        public final PingBuilder setTotalCount(int totalCount) {
            if (totalCount <= 0) {
                throw new IllegalArgumentException("Negative count : " + totalCount);
            }
            this.totalCount = totalCount;
            return this;
        }

        public final int getRequiredReplies() {
            return this.requiredReplies;
        }

        public final PingBuilder setRequiredReplies(int requiredReplies) {
            this.requiredReplies = requiredReplies;
            return this;
        }

        public final boolean isAlive(InetAddress host) throws IOException {
            return this.isAlive(host, 0);
        }

        public final boolean isAlive(InetAddress host, int routingTableIndex) throws IOException {
            return this.execute(host, routingTableIndex).hasRequiredReplies();
        }

        public PingResult execute(InetAddress host) throws IOException {
            return this.execute(host, 0);
        }

        public PingResult execute(InetAddress host, int routingTableIndex) throws IOException {
            return this.platform.ping(host, this, routingTableIndex);
        }
    }

    public static final class PingResult {
        private final int wantedCount;
        private final int repliesCount;
        private final int requiredReplies;
        private final BigDecimal averageRTT;

        private PingResult(int wantedCount, int repliesCount, int requiredReplies, BigDecimal averageRTT) {
            this.wantedCount = wantedCount;
            this.repliesCount = repliesCount;
            this.requiredReplies = requiredReplies;
            this.averageRTT = averageRTT;
        }

        public final int getRepliesCount() {
            return this.repliesCount;
        }

        public final int getRequiredReplies() {
            return this.requiredReplies;
        }

        public final boolean hasRequiredReplies() {
            return this.hasEnoughReplies(this.getRequiredReplies());
        }

        public final boolean hasEnoughReplies(int min) {
            return this.getRepliesCount() >= min;
        }

        public final BigDecimal getAverageRTT() {
            return this.averageRTT;
        }

        public String toString() {
            return String.valueOf(this.getClass().getSimpleName()) + " " + this.getRepliesCount() + "/" + this.wantedCount + " reply(ies), average RTT : " + this.getAverageRTT() + " ms";
        }
    }

    private static abstract class UnixPlatform
    extends Platform {
        private static final Pattern PING_STATS_PATTERN = Pattern.compile("^\\p{Blank}*(?:rtt|round-trip)\\p{Blank}+min/avg/max/(?:mdev|stddev)\\p{Blank}+=\\p{Blank}+([0-9\\.]+)/([0-9\\.]+)/([0-9\\.]+)/([0-9\\.]+)\\p{Blank}+ms$");

        private UnixPlatform() {
        }

        @Override
        public boolean supportsPID() {
            return true;
        }

        @Override
        public final String getPreJava9PID() throws IOException {
            String symlink = this.getSelfProcessSymlink();
            if (symlink == null) {
                return super.getPreJava9PID();
            }
            return Files.readSymbolicLink(Paths.get(symlink, new String[0])).getFileName().toString();
        }

        protected abstract String getSelfProcessSymlink();

        @Override
        public final boolean isRunning(int pid) throws IOException {
            Process p = Runtime.getRuntime().exec(new String[]{"ps", "-p", String.valueOf(pid)});
            return this.exitStatus(p) == 0;
        }

        @Override
        public String getPath(File f) {
            return f.getPath();
        }

        @Override
        protected String getBash() {
            return "bash";
        }

        @Override
        public String readLink(File f) throws IOException {
            Process p = Runtime.getRuntime().exec(new String[]{"readlink", "-n", f.getAbsolutePath()});
            return DesktopEnvironment.cmdSubstitution(p);
        }

        @Override
        public boolean ping(InetAddress host, int timeout) throws IOException {
            try {
                return host.isReachable(timeout);
            }
            catch (SocketException e) {
                Log.get().log(Level.FINER, "Swallow exception", e);
                return false;
            }
        }

        @Override
        protected BigDecimal parsePingAverageRT(String statsLine) {
            Matcher m = PING_STATS_PATTERN.matcher(statsLine);
            if (!m.matches()) {
                throw new IllegalArgumentException("Not matching " + PING_STATS_PATTERN + " :\n" + statsLine);
            }
            return new BigDecimal(m.group(2));
        }

        @Override
        public boolean isAdmin() throws IOException {
            return DesktopEnvironment.cmdSubstitution(this.eval("id -u")).trim().equals("0");
        }
    }
}

