/*
 * Decompiled with CFR 0.152.
 */
package org.openconcerto.utils;

import java.awt.Desktop;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.openconcerto.utils.CollectionUtils;
import org.openconcerto.utils.DesktopEnvironment;
import org.openconcerto.utils.FileUtils;
import org.openconcerto.utils.StringUtils;
import org.openconcerto.utils.io.PercentEncoder;

public abstract class EmailClient {
    private static EmailClient PREFERRED = null;
    private static final Pattern registryPattern = Pattern.compile("\\s+REG_SZ\\s+(.*)$", 8);
    private static final Pattern cmdLinePattern = Pattern.compile("(\"(.*?)\")|([^\\s\"]+)\\b");
    private static final Pattern wsPattern = Pattern.compile("[\\s&&[^ \t]]");
    private static final Pattern dictPattern;
    private static final String AppleMailBundleID = "com.apple.mail";
    private static final String ThunderbirdBundleID = "org.mozilla.thunderbird";
    public static final EmailClient MailTo;
    public static final EmailClient XDG;
    public static final EmailClient Outlook;
    public static final EmailClient AppleMail;
    private final EmailClientType type;

    static {
        String rolePattern = "(?:LSHandlerRoleAll\\s*=\\s*\"([\\w\\.]+)\";\\s*)?";
        dictPattern = Pattern.compile("\\{\\s*(?:LSHandlerRoleAll\\s*=\\s*\"([\\w\\.]+)\";\\s*)?LSHandlerURLScheme = mailto;\\s*(?:LSHandlerRoleAll\\s*=\\s*\"([\\w\\.]+)\";\\s*)?\\}");
        MailTo = new EmailClient(null){

            @Override
            public boolean composeNative(String to, String subject, String body, File ... attachments) {
                return false;
            }
        };
        XDG = new EmailClient(EmailClientType.XDG){

            @Override
            public boolean composeNative(String to, String subject, String body, File ... attachments) throws IOException, InterruptedException {
                ProcessBuilder pb = new ProcessBuilder("xdg-email");
                if (subject != null) {
                    pb.command().add("--subject");
                    pb.command().add(subject);
                }
                if (body != null) {
                    pb.command().add("--body");
                    pb.command().add(body);
                }
                File[] fileArray = attachments;
                int n = attachments.length;
                int n2 = 0;
                while (n2 < n) {
                    File attachment = fileArray[n2];
                    pb.command().add("--attach");
                    pb.command().add(attachment.getAbsolutePath());
                    ++n2;
                }
                pb.command().add(to);
                pb.inheritIO();
                Process process = pb.start();
                process.getOutputStream().close();
                int returnCode = process.waitFor();
                if (returnCode != 0) {
                    throw new IllegalStateException("Non zero return code: " + returnCode);
                }
                return true;
            }
        };
        Outlook = new EmailClient(EmailClientType.Outlook){

            @Override
            protected boolean composeNative(String to, String subject, String body, File ... attachments) throws IOException, InterruptedException {
                DesktopEnvironment de = DesktopEnvironment.getDE();
                File vbs = FileUtils.getFile(EmailClient.class.getResource("OutlookEmail.vbs"));
                ArrayList<String> l = new ArrayList<String>(6);
                l.add("cscript");
                l.add(de.quoteParamForExec(vbs.getAbsolutePath()));
                if (to != null) {
                    l.add(EmailClient.createVBParam("to", to));
                }
                if (subject != null) {
                    l.add(EmailClient.createVBParam("subject", subject));
                }
                l.add(EmailClient.createVBParam("unicodeStdIn", "1"));
                File[] fileArray = attachments;
                int n = attachments.length;
                int n2 = 0;
                while (n2 < n) {
                    File attachment = fileArray[n2];
                    l.add(de.quoteParamForExec(attachment.getAbsolutePath()));
                    ++n2;
                }
                Process process = new ProcessBuilder(l).start();
                BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(process.getOutputStream(), StringUtils.UTF16));
                writer.write(body);
                ((Writer)writer).close();
                int returnCode = process.waitFor();
                if (returnCode != 0) {
                    throw new IllegalStateException("Non zero return code: " + returnCode);
                }
                return true;
            }
        };
        AppleMail = new EmailClient(EmailClientType.AppleMail){

            @Override
            protected boolean composeNative(String to, String subject, String body, File ... attachments) throws IOException, InterruptedException {
                Process process = Runtime.getRuntime().exec(new String[]{"osascript"});
                PrintStream w = new PrintStream(new BufferedOutputStream(process.getOutputStream()));
                w.println("tell application id \"com.apple.mail\"");
                w.println(" set theMessage to make new outgoing message with properties {" + EmailClient.getAppleMailParam(subject, body) + "}");
                if (to != null) {
                    w.println(" tell theMessage to make new to recipient with properties {address:" + StringUtils.doubleQuote(to) + "}");
                }
                File[] fileArray = attachments;
                int n = attachments.length;
                int n2 = 0;
                while (n2 < n) {
                    File attachment = fileArray[n2];
                    w.println(" tell content of theMessage to make new attachment with properties {file name:" + StringUtils.doubleQuote(attachment.getAbsolutePath()) + "} at after last paragraph");
                    ++n2;
                }
                w.println("end tell");
                w.close();
                if (w.checkError()) {
                    throw new IOException();
                }
                int returnCode = process.waitFor();
                if (returnCode != 0) {
                    throw new IllegalStateException("Non zero return code: " + returnCode);
                }
                return true;
            }
        };
    }

    public static final EmailClient getPreferred() throws IOException {
        if (PREFERRED == null) {
            PREFERRED = EmailClient.findPreferred();
            assert (PREFERRED != null);
        }
        return PREFERRED;
    }

    public static final void resetPreferred() {
        PREFERRED = null;
    }

    private static final String createEncodedParam(String name, String value) {
        return String.valueOf(name) + "=" + PercentEncoder.encode(value, StringUtils.UTF8);
    }

    private static final String createASParam(String name, String value) {
        return String.valueOf(name) + ":" + StringUtils.doubleQuote(value);
    }

    private static final String createVBParam(String name, String value) {
        String switchName = "/" + name + ":";
        if (value == null || value.length() == 0) {
            return switchName;
        }
        String encoded = PercentEncoder.encodeUTF16(value);
        assert (encoded.indexOf(34) < 0) : "Encoded contains a double quote, this will confuse cscript";
        return String.valueOf(switchName) + '\"' + encoded + '\"';
    }

    public static final URI getMailToURI(String to, String subject, String body, File ... attachments) throws IOException {
        String encodedTo = to == null ? "" : PercentEncoder.encode(to, StringUtils.UTF8);
        ArrayList<String> l = new ArrayList<String>(4);
        if (subject != null) {
            l.add(EmailClient.createEncodedParam("subject", subject));
        }
        if (body != null) {
            l.add(EmailClient.createEncodedParam("body", body));
        }
        File[] fileArray = attachments;
        int n = attachments.length;
        int n2 = 0;
        while (n2 < n) {
            File attachment = fileArray[n2];
            l.add(EmailClient.createEncodedParam("attachment", attachment.getAbsolutePath()));
            ++n2;
        }
        String query = CollectionUtils.join(l, "&");
        try {
            return new URI("mailto:" + encodedTo + "?" + query);
        }
        catch (URISyntaxException e) {
            throw new IOException("Couldn't create mailto URI", e);
        }
    }

    private static final String getTBParam(String to, String subject, String body, File ... attachments) {
        ArrayList<String> l = new ArrayList<String>(4);
        if (to != null) {
            l.add(EmailClient.createEncodedParam("to", to));
        }
        if (subject != null) {
            l.add(EmailClient.createEncodedParam("subject", subject));
        }
        if (body != null) {
            l.add(EmailClient.createEncodedParam("body", body));
        }
        ArrayList<String> urls = new ArrayList<String>(attachments.length);
        File[] fileArray = attachments;
        int n = attachments.length;
        int n2 = 0;
        while (n2 < n) {
            File attachment = fileArray[n2];
            String rawPath = attachment.toURI().getRawPath();
            String tbURL = String.valueOf(rawPath.startsWith("//") ? "file:///" : "file://") + rawPath;
            urls.add(tbURL);
            ++n2;
        }
        l.add(EmailClient.createEncodedParam("attachment", CollectionUtils.join(urls, ",")));
        return DesktopEnvironment.getDE().quoteParamForExec(CollectionUtils.join(l, ","));
    }

    private static final String getAppleMailParam(String subject, String body) {
        ArrayList<String> l = new ArrayList<String>(3);
        l.add("visible:true");
        if (subject != null) {
            l.add(EmailClient.createASParam("subject", subject));
        }
        if (body != null) {
            l.add(EmailClient.createASParam("content", body));
        }
        return CollectionUtils.join(l, ", ");
    }

    private static String[] tbCommand(String cmdLine, String toReplace, String to, String subject, String body, File ... attachments) {
        String composeArg = EmailClient.getTBParam(to, subject, body, attachments);
        ArrayList<String> arguments = new ArrayList<String>();
        Matcher cmdMatcher = cmdLinePattern.matcher(cmdLine);
        while (cmdMatcher.find()) {
            String quoted = cmdMatcher.group(2);
            String unquoted = cmdMatcher.group(3);
            assert (quoted == null ^ unquoted == null) : "Both quoted and unquoted, or neither quoted nor quoted: " + quoted + " and " + unquoted;
            String arg = quoted != null ? quoted : unquoted;
            boolean replace = arg.equals(toReplace);
            if (replace && !arguments.contains("-compose")) {
                arguments.add("-compose");
            }
            arguments.add(replace ? composeArg : arg);
        }
        return arguments.toArray(new String[arguments.size()]);
    }

    public void compose(String to, String subject, String body, File ... attachments) throws IOException, InterruptedException {
        File[] fileArray = attachments;
        int n = attachments.length;
        int n2 = 0;
        while (n2 < n) {
            File attachment = fileArray[n2];
            if (!attachment.exists()) {
                throw new IOException("Attachment doesn't exist: '" + attachment.getAbsolutePath() + "'");
            }
            ++n2;
        }
        boolean handled = this.composeNative(to, subject = wsPattern.matcher(subject).replaceAll(" "), body, attachments);
        if (!handled) {
            URI mailto = EmailClient.getMailToURI(to, subject, body, attachments);
            Desktop.getDesktop().mail(mailto);
        }
    }

    private static String cmdSubstitution(String ... args) throws IOException {
        return DesktopEnvironment.cmdSubstitution(Runtime.getRuntime().exec(args));
    }

    private static EmailClient findPreferred() throws IOException {
        DesktopEnvironment de = DesktopEnvironment.getDE();
        if (de instanceof DesktopEnvironment.Windows) {
            String out = EmailClient.cmdSubstitution("reg", "query", "HKEY_CLASSES_ROOT\\mailto\\shell\\open\\command");
            Matcher registryMatcher = registryPattern.matcher(out);
            if (registryMatcher.find()) {
                String cmdLine = registryMatcher.group(1);
                if (cmdLine.contains("thunderbird")) {
                    return new ThunderbirdCommandLine(cmdLine, "%1");
                }
                if (cmdLine.toLowerCase().contains("outlook")) {
                    return Outlook;
                }
            }
        } else if (de instanceof DesktopEnvironment.Mac) {
            String bundleID;
            String dict = EmailClient.cmdSubstitution("defaults", "read", "com.apple.LaunchServices", "LSHandlers");
            Matcher dictMatcher = dictPattern.matcher(dict);
            if (dictMatcher.find()) {
                String before = dictMatcher.group(1);
                String after = dictMatcher.group(2);
                assert (before == null ^ after == null) : "Both before and after, or neither before nor after: " + before + " and " + after;
                bundleID = before != null ? before : after;
            } else {
                bundleID = AppleMailBundleID;
            }
            if (bundleID.equals(AppleMailBundleID)) {
                return AppleMail;
            }
            if (bundleID.equals(ThunderbirdBundleID)) {
                File appDir = ((DesktopEnvironment.Mac)de).getAppDir(bundleID);
                File exe = new File(appDir, "Contents/MacOS/thunderbird-bin");
                return new ThunderbirdPath(exe);
            }
        } else {
            if (de instanceof DesktopEnvironment.Gnome) {
                String cmdLine;
                if (de.getVersion().startsWith("2.") && (cmdLine = EmailClient.cmdSubstitution("gconftool", "-g", "/desktop/gnome/url-handlers/mailto/command")).contains("thunderbird")) {
                    return new ThunderbirdCommandLine(cmdLine, "%s");
                }
                return XDG;
            }
            if (de instanceof DesktopEnvironment.KDE) {
                return XDG;
            }
        }
        return MailTo;
    }

    public EmailClient(EmailClientType type) {
        this.type = type;
    }

    public final EmailClientType getType() {
        return this.type;
    }

    protected abstract boolean composeNative(String var1, String var2, String var3, File ... var4) throws IOException, InterruptedException;

    public String toString() {
        EmailClientType t = this.getType();
        return t == null ? "mailto" : t.toString();
    }

    public static final void main(String[] args) throws Exception {
        if (args.length == 1 && "--help".equals(args[0])) {
            System.out.println("Usage: java [-Dparam=value] " + EmailClient.class.getName() + " [EmailClientType args]");
            System.out.println("\tEmailClientType: mailto or " + Arrays.asList(EmailClientType.values()));
            System.out.println("\tparam: to, subject, body, files (seprated by ',' double it to escape)");
            return;
        }
        EmailClient client = EmailClient.createFromString(args);
        System.out.println("Using " + (args.length == 0 ? "preferred" : "passed") + " client : " + client);
        String to = System.getProperty("to", "Pierre Dupond <p.dupond@example.com>, p.dupont@server.com");
        String subject = System.getProperty("subject", "Suj\u00e9 \u20ac du courrier ',to='&;\\<> \"autre'\n2nd line");
        String body = System.getProperty("body", "Bonjour,\n\tsingle ' double \" backslash(arri\u00e8re) \\ slash /");
        String filesPath = System.getProperty("files");
        String[] paths = filesPath == null || filesPath.length() == 0 ? new String[]{} : filesPath.split("(?<!,),(?!,)");
        File[] f = new File[paths.length];
        int i = 0;
        while (i < f.length) {
            f[i] = new File(paths[i].replace(",,", ","));
            ++i;
        }
        client.compose(to, subject, body, f);
    }

    private static final EmailClient createFromString(String ... args) throws IOException {
        if (args.length == 0) {
            return EmailClient.getPreferred();
        }
        if ("mailto".equals(args[0])) {
            return MailTo;
        }
        EmailClientType t = EmailClientType.valueOf(args[0]);
        switch (t) {
            case XDG: {
                return XDG;
            }
            case Outlook: {
                return Outlook;
            }
            case AppleMail: {
                return AppleMail;
            }
            case Thunderbird: {
                Thunderbird res = null;
                if (args.length == 2) {
                    File exe = new File(args[1]);
                    res = Thunderbird.createFromExe(exe);
                    if (res == null) {
                        throw new IOException("Invalid exe : " + exe);
                    }
                } else if (args.length == 3) {
                    res = Thunderbird.createFromCommandLine(args[1], args[2]);
                } else {
                    throw new IllegalArgumentException((Object)((Object)t) + " needs 1 or 2 arguments");
                }
                return res;
            }
        }
        throw new IllegalStateException("Unknown type " + (Object)((Object)t));
    }

    public static enum EmailClientType {
        Thunderbird,
        AppleMail,
        Outlook,
        XDG;

    }

    public static abstract class Thunderbird
    extends EmailClient {
        public static Thunderbird createFromExe(File exe) {
            if (exe == null) {
                throw new NullPointerException();
            }
            if (!exe.isFile()) {
                return null;
            }
            return new ThunderbirdPath(exe);
        }

        public static Thunderbird createFromCommandLine(String cmdLine, String toReplace) {
            return new ThunderbirdCommandLine(cmdLine, toReplace);
        }

        protected Thunderbird() {
            super(EmailClientType.Thunderbird);
        }

        @Override
        public String toString() {
            return this.getClass().getSimpleName();
        }
    }

    private static final class ThunderbirdCommandLine
    extends Thunderbird {
        private final String cmdLine;
        private final String toReplace;

        private ThunderbirdCommandLine(String cmdLine, String toReplace) {
            this.cmdLine = cmdLine;
            this.toReplace = toReplace;
        }

        @Override
        protected boolean composeNative(String to, String subject, String body, File ... attachments) throws IOException {
            Runtime.getRuntime().exec(EmailClient.tbCommand(this.cmdLine, this.toReplace, to, subject, body, attachments));
            return true;
        }

        @Override
        public String toString() {
            return String.valueOf(super.toString()) + " " + this.cmdLine;
        }
    }

    private static final class ThunderbirdPath
    extends Thunderbird {
        private final File exe;

        private ThunderbirdPath(File exe) {
            this.exe = exe;
        }

        @Override
        protected boolean composeNative(String to, String subject, String body, File ... attachments) throws IOException {
            String composeArg = EmailClient.getTBParam(to, subject, body, attachments);
            Runtime.getRuntime().exec(new String[]{this.exe.getPath(), "-compose", composeArg});
            return true;
        }

        @Override
        public String toString() {
            return String.valueOf(super.toString()) + " " + this.exe;
        }
    }
}

